// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
/**
 * Module used for handling discovering game servers on the LAN.
 *
 * This is implemented by having the client send out UDP datagrams to a
 * broadcast or multicast address on the local network, which any listening game
 * server would send back a response containing information about it.
 *
 * Here is the protocol:
 *
 * The client sends the message `DISCOVER_MAGIC_NUMBER` to the server as a UDP
 * datagram. The server, listening on UDP port `SERVER_PORT` (right now 4081),
 * ensures that the exact message was sent by the client and sends a unicast
 * response of the server's information back to the client.
 *
 * This response back to the client is:
 * 	 1. Magic number of `RESPONSE_MAGIC_NUMBER`.
 * 	 2. Vu30 (see net/serp/vu30.rs) for the LAN discovery protocol id.
 * 	 3. Vu30 for the Play request type supported. Need to specify this as
 * 	    forks of the game might use a different request for playing.
 * 	 4. Vu30 for the version of that play request.
 * 	 5. 16 bits for game port number.
 * 	 6. 16 bits for current number of players.
 * 	 7. 16 bits for maximum number of players. This value can be between and
 * 	    including 1 – 65535, or 0 which will be represented as there being no
 * 	    limit.
 * 	 8. Null-terminated UTF-8 string for the game id. Can be empty. If there
 * 	    is no null terminator, the remaining fields are empty.
 * 	 9. Null-terminated UTF-8 string for the name. Can be empty. If there is
 * 	    no null terminator, then there is no description.
 * 	10. UTF-8 string for the description (not null-terminated). Can also be
 * 	    empty.
 *
 * Now some notes about this:
 *
 * 1. Just like with the client's message, if the server's message doesn't have
 *    the magic number match, the datagram is dropped.
 * 2. The integers in 2 – 4 are unsigned and little-endian.
 * 3. If the strings aren't UTF-8, the datagram is dropped.
 * 4. Current players > max players is allowed and the datagram won't be
 *    dropped.
 * 5. There is a 512 byte limit to the response. Servers won't send responses
 *    larger than this and clients will drop all datagrams violating this.
 *
 *    This exact value of 512 bytes is arbitrary but there needs to be some
 *    limit, as LAN discovery might not won't work on IPv6 if the data is >1280
 *    bytes, and I need to take into account the possibility of higher level
 *    protocols (like those in VPNs) wrapping the data.
 *
 *    IPv4 isn't a problem as the limit for UDP is >65 kB, but there will be a
 *    lot of fragmentation which would decrease the probability of successfully
 *    delivering the message, as every single fragment needs to be correctly
 *    delivered.
 */
#[cfg(feature = "client")] pub mod client;
pub(super) mod server;
pub(super) mod security;

use std::num::NonZeroU16;
use std::str;

use super::server::PORT;
use super::serp::vu30::{Vu30, ExtendVu30};

pub(in crate::net) const PROTOCOL_VERSION: Vu30 = Vu30::ZERO;
const DISCOVER_MAGIC_NUMBER: &[u8] = b"spsh-discover";
const RESPONSE_MAGIC_NUMBER: &[u8] = b"spsh-response";

/*
 * Port for the server to listen on, and the preferred port for the client to
 * listen on. These are both UDP.
 *
 * The purpose of having the client listen on a preferred port is so a firewall
 * rule on the client can be specified to allow incoming traffic for LAN
 * discovery, rather than requiring the firewall to allow all ports.
 */
const SERVER_PORT: u16 = PORT + 1;
#[cfg(feature = "client")] const CLIENT_PORT: u16 = PORT + 2;

/**
 * The limit of the size of a response. Servers won't send responses larger than
 * this and clients will drop all datagrams violating this.
 *
 * This exact value of 512 bytes is arbitrary but there needs to be some limit,
 * as LAN discovery might not won't work on IPv6 if the data is >1280 bytes, and
 * I need to take into account the possibility of higher level protocols (like
 * those in VPNs) wrapping the data.
 *
 * IPv4 isn't a problem as the limit for UDP is >65 kB, but there will be a lot
 * of fragmentation which would decrease the probability of successfully
 * delivering the message, as every single fragment needs to be correctly
 * delivered.
 *
 * Overall, 512 is a nice number and should be sufficient so I'm using this.
 */
const MAX_LEN: usize = 512;

// Message the server sends back to the client
#[derive(Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct LanServerInfo {
	pub version: Vu30,
	pub request_type: Vu30,
	pub request_version: Vu30,
	pub port: u16,
	pub cur_players: u16,
	pub max_players: Option<NonZeroU16>,
	pub game_id: Box<str>,
	pub name: Box<str>,
	pub desc: Box<str>,
}

impl LanServerInfo {
	pub fn validate(self) -> Result<LanServerInfo, String> {
		let has_null_byte = |s: &str| s.as_bytes().iter().any(|&x| x == 0);

		if has_null_byte(&self.game_id) { return Err(String::from("game id contains a null byte")); }
		if has_null_byte(&self.name) { return Err(String::from("name contains a null byte")); }

		let len = self.get_len();
		if len <= MAX_LEN {
			Ok(self)
		} else {
			Err(format!("game id, name and description are too long, resulting in a datagram size of {len} bytes, expected at most {MAX_LEN} bytes"))
		}
	}

	#[cfg(feature = "client")]
	pub fn format_players(&self) -> String {
		match (self.cur_players, self.max_players) {
			(cur, Some(max)) if max.get() == 1 => format!("{cur}/1 player"),
			(cur, Some(max)) => format!("{cur}/{max} players"),
			(1, None) => String::from("1 player"),
			(cur, None) => format!("{cur} players"),
		}
	}

	pub fn with_cur_players(mut self, cur_players: u16) -> LanServerInfo {
		self.cur_players = cur_players;
		self
	}

	pub fn serialise(&self) -> Vec<u8> {
		let mut ret = Vec::with_capacity(self.get_len());
		ret.extend_from_slice(RESPONSE_MAGIC_NUMBER);
		ret.extend_vu30(self.version);
		ret.extend_vu30(self.request_type);
		ret.extend_vu30(self.request_version);
		ret.extend_from_slice(&self.port.to_le_bytes());
		ret.extend_from_slice(&self.cur_players.to_le_bytes());
		ret.extend_from_slice(&self.max_players.map_or(0, NonZeroU16::get).to_le_bytes());
		ret.extend_from_slice(self.game_id.as_bytes());
		ret.push(0);
		ret.extend_from_slice(self.name.as_bytes());
		ret.push(0);
		ret.extend_from_slice(self.desc.as_bytes());
		ret
	}

	fn get_len(&self) -> usize {
		RESPONSE_MAGIC_NUMBER.len() + // Magic number
		self.version.len() + self.request_type.len() + self.request_version.len() + // Protocol fields
		2 /* u16 size */ * 3 /* port, cur_players, max_players */ + // Integral fields
		self.game_id.len() + 1 + self.name.len() + 1 + self.desc.len() // Game id, name, description and null terminators
	}
}
