// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod networked;
#[cfg(feature = "client")] pub mod singleplayer;

use super::{message_stream::DynMs, serp::PlayRequest};

// Can be sent between threads
pub(super) type DynConnLis = dyn ConnectionListener + Send;
pub(super) type DynIncomingConnection = dyn IncomingConnection + Send;

/**
 * A listener for incoming connections.
 *
 * This trait is used by the server to accept incoming connections.
 */
pub trait ConnectionListener {
	/**
	 * Returns a pending connection, if it exists.
	 *
	 * On success, if there's a pending connection returns an
	 * `IncomingConnection` in the Ok(Some(...)) variant for a client attempting
	 * to connect. This connection can be accepted to return a `MessageStream` or
	 * rejected with a provided reason. Returns Ok(None) if there's no pending
	 * connection.
	 *
	 * If the server should close for whatever reason, the Err variant should be
	 * returned.
	 *
	 * This method is non-blocking.
	 */
	fn get(&mut self) -> Result<Option<(Box<DynIncomingConnection>, PlayRequest)>, ()>;
}

/**
 * A connection attempt made by the client. Can be accepted to return a message
 * stream or rejected with a provided reason.
 */
pub trait IncomingConnection {
	fn accept(self: Box<Self>) -> Result<Box<DynMs>, (String, String)>;
	fn reject(self: Box<Self>, reason: String);
}
