// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
#[cfg(feature = "client")] mod game;
#[cfg(feature = "client")] mod playing;
mod net;
mod world;
mod utils;
mod blocks;
#[cfg(feature = "client")] mod gui;

use std::{env, process, io::Write};
#[cfg(feature = "client")] use std::sync::Arc;

use clap::{Parser, Args, Subcommand, ValueEnum};
use log::{Level, LevelFilter};
use env_logger::{Env, Builder};
use rustls::crypto::aws_lc_rs;
use colored::Colorize;

use net::server::{self, PORT};
use utils::escape;
#[cfg(feature = "client")] use game::Game;
#[cfg(feature = "client")] use net::{Address, info::Info};
#[cfg(feature = "client")] use world::{colour::Colour, player::PlayerName};

// Heavily based on clap documentation: https://docs.rs/clap/latest/clap/_derive/_tutorial/chapter_0/index.html
#[derive(Parser)]
#[command(version, about, long_about = None)]
struct Cli {
	#[cfg(feature = "client")] #[command(subcommand)] command: Option<Command>,
	#[cfg(not(feature = "client"))] #[command(subcommand)] command: Command,
}

#[derive(Subcommand)]
enum Command {
	/// Starts a game immediately rather than opening the GUI
	#[cfg(feature = "client")]
	Play {
		/// The address of the server
		#[arg(default_value_t = Box::from("127.0.0.1"), conflicts_with = "singleplayer")] host: Box<str>,

		/// The game id of the server to join
		#[arg(default_value_t = Box::from(""), conflicts_with = "singleplayer")] game_id: Box<str>,

		/// The UDP port of the server
		#[arg(short, long, default_value_t = PORT, conflicts_with = "singleplayer")] port: u16,

		/// Plays singleplayer rather than multiplayer
		#[arg(short, long)] singleplayer: bool,

		/// Requests to join as a spectator
		#[arg(short = 'e', long)] spectate: bool,

		/// Joins the game with this name
		#[arg(short, long)] name: Option<Arc<str>>,

		/// Joins the game with this colour
		#[arg(short, long)] colour: Option<Box<str>>,

		/// Whether to check sanity checks sent by the server
		#[arg(long)] sanity_checks: bool,
	},

	/// Requests build information about a server or the client
	#[cfg(feature = "client")]
	Info {
		/// The address of the server, skip this to print info about the client
		host: Option<Box<str>>,

		/// The UDP port of the server
		#[arg(short, long, default_value_t = PORT, requires = "host")] port: u16,
	},

	/// Starts a server
	Server(ServerArgs),

	/// Saves the block map generated from a configuration file to an image
	SaveBlocks {
		/// Path of the input configuration file
		#[arg(id = "CONFIG", short = 'c', long = "config")] config_path: String,

		/// Path of the output image file of the blocks in the world
		#[arg(id = "OUTPUT", short = 'o', long = "output")] output_path: String,
	},
}

#[derive(Args)]
struct ServerArgs {
	/// List of game servers to start
	game_servers: Vec<String>,

	/// The address to bind on
	#[arg(id = "ADDRESS", short = 'a', long = "address", default_value_t = String::from("::"))] addr: String,

	/// The UDP port to listen to
	#[arg(short, long, default_value_t = PORT)] port: u16,

	/// The path of the certificate chain in PEM format
	#[arg(long, requires = "key_path")] cert_path: Option<Box<str>>,

	/// The path of the private key in PEM format
	#[arg(long, requires = "cert_path")] key_path: Option<Box<str>>,

	/// Whether to periodically send copies of the world to allow clients to perform sanity checks
	#[arg(long, default_value_t = false)] sanity_checks: bool,

	/// Whether LAN discovery is enabled and its visibility
	#[arg(value_enum, long, default_value_t = LanDiscoveryConfig::Local)] lan_discovery: LanDiscoveryConfig,
}

#[derive(Clone, Copy, PartialEq, Eq, ValueEnum)]
enum LanDiscoveryConfig {
	/// LAN discovery is disabled
	Disabled,

	/// Only responds to LAN discovery messages on the local network (recommended)
	Local,

	/// Responds to LAN discovery messages regardless of their source. If the
	/// game server is accessible on the internet, then discovery of all running
	/// game servers is possible by anyone on the internet. Because of a possible
	/// desire to use private servers with a secret game id, this option is NOT
	/// recommended, as the game ids of all running servers will be exposed.
	Global,
}

fn main() {
	let command = if cfg!(feature = "client") {
		Cli::parse().command
	} else {
		Cli::try_parse().unwrap_or_else(|err| {
			let _ = err.print();
			let server_only_note = "\nNOTE: Using the server-only build. Some features are disabled.";
			eprintln!("{}", server_only_note.yellow().bold());

			process::exit(err.exit_code());
		}).command
	};
	let mut builder = Builder::from_env(Env::default().default_filter_or("spaceships=info"));
	builder.format_timestamp_millis();
	if env::var("PERF_LOG").is_ok() {
		builder.filter(Some("perf"), LevelFilter::Trace);
		builder.format(|buf, record| if record.target() == "perf" {
			writeln!(buf, "{}", record.args())
		} else { Ok(()) });
	} else {
		// Reproduces the same log format used by env_logger but with escaping
		builder.format(|buf, record| {
			let body = escape::singleline(&record.args().to_string());
			let level = match record.level() {
				Level::Error => "ERROR".red().bold(),
				Level::Warn => "WARN".yellow(),
				Level::Info => "INFO".green(),
				Level::Debug => "DEBUG".blue(),
				Level::Trace => "TRACE".cyan(),
			};
			let open = "[".bright_black();
			let close = "]".bright_black();
			writeln!(buf, "{open}{} {level}  {}{close} {body}", buf.timestamp_millis(), record.module_path().unwrap_or_default())
		});
	}
	builder.init();

	/*
	 * Not sure if to use ring or aws_lc_rs, but Rustls uses aws_lc_rs as the
	 * default, so I will use it as well. A bit of research suggests that
	 * aws_lc_rs is slightly faster.
	 *
	 * Source for this bit of code:
	 * https://github.com/rustls/rustls/issues/1938#issuecomment-2099920256
	 */
	aws_lc_rs::default_provider().install_default().unwrap();

	if let Err(err) = run(command) {
		log::error!("{err}");
		process::exit(1);
	}
}

#[cfg(feature = "client")]
fn run(command: Option<Command>) -> Result<(), String> {
	fn convert_colour(colour: Option<Box<str>>) -> Result<Option<Colour>, String> {
		match colour.map(|s| Colour::try_from(s.as_ref())) {
			Some(Ok(colour)) => Ok(Some(colour)),
			Some(Err(err)) => Err(format!("invalid colour: {err}")),
			None => Ok(None),
		}
	}

	fn convert_name(name: Option<Arc<str>>) -> Result<Option<PlayerName>, String> {
		match name.map(PlayerName::try_from) {
			Some(Ok(name)) => Ok(Some(name)),
			Some(Err(err)) => Err(format!("invalid name: {err}")),
			None => Ok(None),
		}
	}

	match command {
		Some(Command::Play { singleplayer: true, sanity_checks, spectate, name, colour, .. }) => Game::singleplayer(spectate, convert_name(name)?, convert_colour(colour)?, sanity_checks)?,
		Some(Command::Play { singleplayer: false, host, game_id, port, spectate, name, colour, sanity_checks }) => {
			let addr = Address { host, port };
			Game::multiplayer(addr, game_id, spectate, convert_name(name)?, convert_colour(colour)?, sanity_checks).map_err(|err| format!("failed starting multiplayer: {err}"))?;
		},
		Some(Command::Info { host, port }) => {
			if let Some(host) = host {
				match Info::request(host.as_ref(), port) {
					Ok(info) => info.print(),
					Err(err) => return Err(format!("failed receiving server info: {err}")),
				}
			} else {
				Info::new().print();
			}
		},
		Some(Command::Server(args)) => server::run(args)?,
		Some(Command::SaveBlocks { config_path, output_path }) => server::save_blocks(&config_path, &output_path)?,
		None => Game::title()?,
	}

	Ok(())
}

#[cfg(not(feature = "client"))]
fn run(command: Command) -> Result<(), String> {
	match command {
		Command::Server(args) => server::run(args)?,
		Command::SaveBlocks { config_path, output_path } => server::save_blocks(&config_path, &output_path)?,
	}

	Ok(())
}
