// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::any::Any;

use glium::Frame;
use winit::event::Event;

use crate::game::{Game, config::action::ActionEvent, state::{GameState, GameStateId, Status}};
use crate::gui::{GuiState, style::UiExt};

pub struct Title {
	gui_state: GuiState,
}

impl Title {
	pub fn new() -> Title {
		Title {
			gui_state: GuiState::new(),
		}
	}
}

impl GameState for Title {
	fn enable(&mut self, game: &mut Game) {
		self.gui_state.enable(&mut game.gui);
	}

	fn push(&mut self, _game: &mut Game, msg: Option<Box<dyn Any>>) {
		debug_assert!(msg.is_none());
	}

	fn disable(&mut self, _game: &mut Game) {}

	fn action(&mut self, action: ActionEvent) {
		self.gui_state.action(action);
	}

	fn event(&mut self, game: &mut Game, event: &Event<()>) {
		self.gui_state.event(game, event);
	}

	fn loop_iter(&mut self, game: &mut Game, frame: &mut Frame) -> Status {
		enum Action {
			PlaySelection,
			Settings,
			About,
		}

		let mut action = None;

		GuiState::update(&mut game.gui, "title", &game.window, |_ctx, ui| {
			ui.vertical_centered(|ui| {
				ui.h1("Spaceships").add();
				if ui.b1("Play").clicked() {
					action = Some(Action::PlaySelection);
				}
				ui.add_space(4.0);
				if ui.b1("Settings").clicked() {
					action = Some(Action::Settings);
				}
				ui.add_space(4.0);
				if ui.b1("About").clicked() {
					action = Some(Action::About);
				}
				ui.add_space(4.0);

				if ui.b1("Exit").clicked() {
					self.gui_state.exit();
				}
			});
		});

		game.gui.render(&game.display, frame);

		match action {
			_ if self.gui_state.should_exit() => Status::PopState,
			Some(Action::PlaySelection) => Status::PushState(GameStateId::PlaySelection, None),
			Some(Action::Settings) => Status::PushState(GameStateId::Settings, None),
			Some(Action::About) => Status::PushState(GameStateId::About, None),
			None => Status::Ok,
		}
	}
}
