// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::time::{Instant, Duration};

use egui::{CentralPanel, ScrollArea, Ui, Context};
use winit::{window::Window, event::{Event, WindowEvent}};

use super::Gui;

use crate::game::{Game, config::action::{Action, ActionEvent, ActionState}};

pub struct GuiState {
	last_repaint: Instant,
	should_exit: bool,
}

impl GuiState {
	pub fn new() -> GuiState {
		GuiState {
			last_repaint: Instant::now(),
			should_exit: false,
		}
	}

	pub fn enable(&mut self, gui: &mut Gui) {
		self.should_exit = false;
		self.should_repaint();
		gui.do_extra_pass();
	}

	pub fn action(&mut self, action: ActionEvent) {
		if matches!(action, ActionEvent::Action(Action::GuiLeave, ActionState::Pressed)) {
			self.should_exit = true;
		}
	}

	pub fn event(&mut self, game: &mut Game, event: &Event<()>) {
		match event {
			Event::WindowEvent { event, .. } => {
				/*
				 * Don't repaint when a RedrawRequested is emitted, as those redraw
				 * requests can come from the below call to request_redraw, rather
				 * than any actual need to redraw.
				 */
				if (game.gui.inner.on_event(&game.window, event).repaint && *event != WindowEvent::RedrawRequested) || matches!(event, WindowEvent::MouseInput { .. }) {
					self.should_repaint();
					game.window.request_redraw();
				}
			}
			Event::AboutToWait => {
				// Waits for egui animations to finish
				if self.last_repaint.elapsed() < Duration::from_millis(500) {
					game.window.request_redraw();
				}
			}
			_ => ()
		}
	}

	pub fn update(gui: &mut Gui, id: &str, window: &Window, mut add_contents: impl FnMut(&Context, &mut Ui)) {
		gui.update(window, |ctx| {
			CentralPanel::default().show(ctx, |ui| {
				ScrollArea::vertical().id_salt(id).show(ui, |ui| {
					ui.vertical_centered(|ui| {
						// Added through h1
						//ui.add_space(60.0);
						add_contents(ctx, ui);
						ui.add_space(60.0);
					});
				});
			});
		});
	}

	pub fn should_repaint(&mut self) {
		self.last_repaint = Instant::now();
	}

	pub fn should_exit(&mut self) -> bool {
		let ret = self.should_exit;
		self.should_exit = false;
		ret
	}

	pub fn exit(&mut self) {
		self.should_exit = true;
	}
}
