// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::{collections::{BTreeMap, BTreeSet}, net::{IpAddr, SocketAddr}, fmt::Write, time::Instant};

use egui::{RichText, TextStyle, Color32, Frame, Margin, FontId, FontFamily, SelectableLabel};
use glium::Frame as GliumFrame;

use super::common::{Common, CommonGs};

use crate::game::{Game, config};
use crate::gui::{GuiState, LABEL_SIZE, style::UiExt};
use crate::net::{Address, lan_discovery::{LanServerInfo, client::LanDiscoveryClient}, serp::request::{PlayRequest, PLAY_VERSION, PLAY_VERSION_FIRST_DEV, request_types}, utils::SocketAddrToCanonical};

struct LanServerEntry {
	info: LanServerInfo,
	socket_addrs: Vec<SocketAddr>,
	selected_index: usize,
	manually_selected: bool,
	init_time: Instant,
}

impl LanServerEntry {
	fn new(info: LanServerInfo, socket_addrs: Vec<SocketAddr>, init_time: Instant) -> LanServerEntry {
		LanServerEntry { info, socket_addrs, selected_index: 0, manually_selected: false, init_time }
	}
}

pub struct PsLocal {
	common: Common,
	lan_discovery: LanDiscoveryClient,
	servers: BTreeMap<LanServerInfo, (u16, BTreeSet<SocketAddr>, Instant)>,
	server_list: Vec<LanServerEntry>,
}

const FADE_IN_TIME: f32 = 0.125;

impl PsLocal {
	pub fn new() -> PsLocal {
		PsLocal {
			common: Common::new(),
			lan_discovery: LanDiscoveryClient::new(),
			servers: BTreeMap::new(),
			server_list: Vec::new(),
		}
	}

	fn update_lan_servers(&mut self, now: Instant) {
		self.lan_discovery.update();

		let mut changed = false;
		while let Some((info, socket_addr)) = self.lan_discovery.get() {
			let server = self.servers.entry(info.clone().with_cur_players(0)).or_insert_with(|| (info.cur_players, BTreeSet::new(), now));
			changed |= info.cur_players != server.0;
			server.0 = info.cur_players;
			changed |= server.1.insert(socket_addr);
		}

		if changed {
			/*
			 * Stores the previously selected address to avoid it suddenly changing
			 * from a new response arriving and the player possibly joining a
			 * server they didn't intend on, if they explicitly selected an
			 * address.
			 *
			 * Calling `filter_map` and using an `if let Some(...)` to make things
			 * feel safer, even though I don't think they can ever fail if I used
			 * `unwrap` instead.
			 */
			let prev_selected: BTreeMap<LanServerInfo, SocketAddr> = self.server_list.iter().filter_map(|entry|
				entry.manually_selected.then(|| entry.socket_addrs.get(entry.selected_index).map(|&addr| (entry.info.clone().with_cur_players(0), addr))).flatten()
			).collect();

			self.server_list.clear();
			self.server_list.extend(self.servers.iter().map(|(info, (cur_players, socket_addrs, init_time))| {
				let mut server = LanServerEntry::new(info.clone().with_cur_players(*cur_players), socket_addrs.iter().map(|addr| addr.to_canonical()).collect(), *init_time);
				if let Some(addr) = prev_selected.get(info) {
					server.selected_index = server.socket_addrs.iter().position(|a| a == addr).unwrap_or_default();
				}
				server
			}));
			self.server_list.sort_by(|a, b| a.info.cur_players.cmp(&b.info.cur_players).reverse().then_with(|| a.info.name.cmp(&b.info.name)));

			for socket_addrs in self.server_list.iter_mut().map(|server| &mut server.socket_addrs) {
				socket_addrs.sort_by_key(|addr| {
					// Priority kind of arbitrary and based on personal preference
					match addr.ip() {
						_ if addr.ip().is_loopback() => 1,
						IpAddr::V4(addr) if addr.is_link_local() => 3,
						IpAddr::V6(addr) if addr.segments()[0] == 0xfe80 => 3,
						_ => 2,
					}
				});
			}
		}
	}

	fn socket_addr_without_port(socket_addr: SocketAddr) -> String {
		match socket_addr {
			SocketAddr::V4(addr) => addr.ip().to_string(),
			SocketAddr::V6(addr) => {
				let mut addr_str = addr.ip().to_string();
				if addr.scope_id() != 0 { // A scope id of zero in the socket address doesn't get printed
					let _ = write!(addr_str, "%{}", addr.scope_id());
				}
				addr_str
			}
		}
	}
}

impl CommonGs for PsLocal {
	fn common(&mut self) -> &mut Common { &mut self.common }

	fn disable(&mut self) {
		self.lan_discovery.stop();
	}

	fn loop_iter(&mut self, game: &mut Game, frame: &mut GliumFrame) -> bool {
		const SMALL_COLOUR: Color32 = Color32::from_gray(0xaf);

		let config = &mut game.config.borrow_mut();

		let now = Instant::now();
		self.update_lan_servers(now);

		GuiState::update(&mut game.gui, "ps-local", &game.window, |ctx, ui| {
			ui.h1("Local Servers").add();

			let enabled = !self.lan_discovery.open();
			if ui.b2_enabled(if enabled { "Search LAN" } else { "Searching..." }, enabled).clicked() {
				// Clears these as previous servers might be outdated
				self.servers.clear();
				self.server_list.clear();

				if let Err(err) = self.lan_discovery.discover() {
					log::warn!("failed searching LAN: {err}");
				}
			}

			ui.add_space(30.0);
			config::set_changed!(config, ui.checkbox(&mut config.gui.spectate, "Join as Spectator").changed());
			ui.add_space(30.0);

			Frame::none().outer_margin(Margin::symmetric(80.0, 0.0)).show(ui, |ui| {
				for server in &mut self.server_list {
					// Provides a message if the LAN server is unsupported
					let unsupported_message = if server.info.request_type != request_types::PLAY {
						Some(format!("Game server is unsupported. Check that this game is running the same version as the server. It is possible that the server is running a different fork. Technical details: client request type = {}, server request type = {}.", request_types::PLAY, server.info.request_type))
					} else if server.info.request_version != PLAY_VERSION {
						let mut msg = String::new();
						if server.info.request_version.get() >= PLAY_VERSION_FIRST_DEV {
							let _ = write!(msg, "Game server is running a development version of the game which isn't supported.");
						} else {
							let (older_or_newer, older, newer) = if server.info.request_version < PLAY_VERSION {
								("an old", "server", "client")
							} else {
								("a new", "client", "server")
							};
							let _ = write!(msg, "Game server is running {older_or_newer}er version of the game which isn't supported. To play the game, update the {older} to match the {newer}'s version or revert the {newer} to the {older}'s version.");
						}

						let _ = write!(msg, " Technical details: server version = {}, client version = {}.", server.info.request_version, PLAY_VERSION);
						Some(msg)
					} else {
						None
					};

					let background = if unsupported_message.is_some() { Color32::from_rgb(0x3f, 0x27, 0x27) } else { Color32::from_gray(0x27) };
					let supported = unsupported_message.is_none();
					let max_opacity = if supported { 1.0 } else { 0.5 };

					ui.set_opacity((now.duration_since(server.init_time).as_secs_f32() / FADE_IN_TIME).min(1.0) * max_opacity);

					let response = Frame::none().fill(background).inner_margin(Margin::same(8.0)).show(ui, |ui| {
						let mut name = server.info.name.as_ref();
						if name.trim().is_empty() {
							name = "Unnamed Server";
						}

						ui.h2(name).no_padding().selectable().add();
						ui.add_space(2.0);
						if !server.info.desc.trim().is_empty() {
							ui.h3(server.info.desc.as_ref()).no_padding().selectable().add();
						}
						ui.add_space(12.0);

						ui.horizontal_wrapped(|ui| {
							let size = ui.spacing().button_padding.y * 2.0 + LABEL_SIZE;
							ui.label(RichText::new(format!("{}, at", server.info.format_players())).color(SMALL_COLOUR).font(FontId::new(size, FontFamily::Proportional)));

							let mut new_selected = None;
							for (i, &addr) in server.socket_addrs.iter().enumerate() {
								if ui.add_enabled(supported, SelectableLabel::new(i == server.selected_index, RichText::new(PsLocal::socket_addr_without_port(addr)).text_style(TextStyle::Body))).clicked() {
									new_selected = Some(i);
								}
							}

							let game_id_str = if server.info.game_id.is_empty() {
								String::from("empty game id")
							} else {
								format!("game id \"{}\"", server.info.game_id)
							};

							ui.label(RichText::new(format!("with {game_id_str} on port {}", server.info.port)).color(SMALL_COLOUR).font(FontId::new(size, FontFamily::Proportional)));

							if let Some(i) = new_selected {
								server.selected_index = i;
								server.manually_selected = true;
							}
						});

						ui.add_space(8.0);

						if let Some(starter) = self.common.connection_starter() {
							if ui.b2_enabled("Play", supported).clicked() {
								let addr = Address {
									host: PsLocal::socket_addr_without_port(server.socket_addrs[server.selected_index]).into_boxed_str(),
									port: server.info.port,
								};

								let request = PlayRequest {
									game_id: server.info.game_id.clone(),
									spectate: config.gui.spectate,
									style: config.create_style(),
									extension: Box::new([]),
								};

								starter.connect(game.endpoint.get(&game.tokio_runtime), &game.tokio_runtime, addr, request);
							}
						} else {
							ui.b2_enabled("Play", false);
						}
					}).response;

					if let Some(msg) = unsupported_message {
						response.on_hover_ui_at_pointer(|ui| { ui.label(msg); });
					}

					ui.set_opacity(1.0);
				}

				let mut rect = ui.max_rect();
				rect.set_height((rect.height() - 200.0).max(400.0));
				ui.expand_to_include_rect(rect);
			});

			self.common.finish_update(ctx, ui);
		});

		game.gui.render(&game.display, frame);

		self.lan_discovery.open()
	}
}
