// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use egui::{Color32, TextEdit};
use glium::Frame as GliumFrame;

use super::common::{Common, CommonGs};

use crate::game::{Game, config::{self, Config}};
use crate::gui::{GuiState, style::{UiExt, FontSize, H4_SIZE}};
use crate::net::{Address, serp::PlayRequest};

pub struct PsEnterAddress {
	common: Common,
	port: String,
	prev_was_u16: bool,
}

impl PsEnterAddress {
	pub fn new(config: &Config) -> PsEnterAddress {
		PsEnterAddress {
			common: Common::new(),
			port: config.gui.port.to_string(),
			prev_was_u16: true,
		}
	}
}

impl CommonGs for PsEnterAddress {
	fn common(&mut self) -> &mut Common { &mut self.common }

	fn loop_iter(&mut self, game: &mut Game, frame: &mut GliumFrame) -> bool {
		let config = &mut game.config.borrow_mut();

		GuiState::update(&mut game.gui, "ps-enter-address", &game.window, |ctx, ui| {
			ui.h1("Enter Address").no_bottom_padding().add();

			ui.h3("Host").add();
			config::set_changed!(config, ui.add(TextEdit::singleline(&mut config.gui.host).font_size(H4_SIZE)).changed());

			ui.h3("Game ID").add();
			config::set_changed!(config, ui.add(TextEdit::singleline(&mut config.gui.game_id).font_size(H4_SIZE)).changed());

			ui.h3("Port").add();

			let style = ui.style_mut();
			let orig_bg = style.visuals.extreme_bg_color;
			if !self.prev_was_u16 {
				style.visuals.extreme_bg_color = Color32::from_rgb(0x5f, 0x00, 0x00);
			}
			ui.add(TextEdit::singleline(&mut self.port).font_size(H4_SIZE));
			ui.style_mut().visuals.extreme_bg_color = orig_bg;
			let port = self.port.parse().ok();
			self.prev_was_u16 = port.is_some();

			ui.h3("Options").add();
			config::set_changed!(config, ui.checkbox(&mut config.gui.spectate, "Join as Spectator").changed());

			ui.add_space(30.0);

			if let (Some(starter), Some(port)) = (self.common.connection_starter(), port) {
				if config.gui.port != port {
					config.changed();
				}

				config.gui.port = port;

				if ui.b2("Play").clicked() {
					let addr = Address {
						host: Box::from(config.gui.host.as_str()),
						port,
					};

					let request = PlayRequest {
						game_id: Box::from(config.gui.game_id.as_str()),
						spectate: config.gui.spectate,
						style: config.create_style(),
						extension: Box::new([]),
					};

					starter.connect(game.endpoint.get(&game.tokio_runtime), &game.tokio_runtime, addr, request);
				}
			} else {
				ui.b2_enabled("Play", false);
			}

			self.common.finish_update(ctx, ui);
		});

		game.gui.render(&game.display, frame);

		false
	}
}
