// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::any::Any;

use glium::Frame as GliumFrame;
use winit::event::Event;
use egui::{Context, Ui, Window, Id, Align2};
use tokio::{runtime::Runtime, task::JoinHandle};
use quinn::Endpoint;

use crate::net::{Address, client::{Client, ClientRes}, serp::PlayRequest};
use crate::game::{Game, config::action::ActionEvent, state::{GameState, GameStateId, Status}};
use crate::gui::{GuiState, style::UiExt};

/// Common components shared by the PlaySelection sub-states.
pub(super) struct Common {
	gui_state: GuiState,
	connection_task: Option<JoinHandle<Result<ClientRes, String>>>,
	popup_error: Option<Box<str>>,
}

pub(super) struct ConnectionStarter<'a>(&'a mut Common);

impl ConnectionStarter<'_> {
	pub fn connect(self, endpoint: Result<Endpoint, String>, tokio_runtime: &Runtime, addr: Address, request: PlayRequest) {
		debug_assert!(self.0.connection_task.is_none());
		self.0.connection_task = Some(tokio_runtime.spawn(Client::multiplayer(endpoint, addr, request, false))); // Too lazy to write code that passes whether to have sanity checks to this
	}
}

impl Common {
	pub fn new() -> Common {
		Common {
			gui_state: GuiState::new(),
			connection_task: None,
			popup_error: None,
		}
	}

	/**
	 * If a connection isn't already attempted, returns a handle that allows a
	 * connection to be started. Otherwise returns `None` if a connection is
	 * being waited for.
	 */
	pub fn connection_starter(&mut self) -> Option<ConnectionStarter> {
		self.connection_task.is_none().then_some(ConnectionStarter(self))
	}

	pub fn is_connecting(&self) -> bool {
		self.connection_task.is_some()
	}

	/**
	 * Returns the result of the attempted connection if the connection task
	 * exists and it has finished. Otherwise returns `None`.
	 */
	fn connection_finished(&mut self, tokio_runtime: &Runtime) -> Option<Result<ClientRes, String>> {
		if let Some(task) = self.connection_task.take() {
			if !task.is_finished() {
				self.connection_task = Some(task);
				None
			} else {
				Some(tokio_runtime.block_on(task).unwrap_or_else(|err| Err(format!("failed completing connection task: {err}"))))
			}
		} else { None }
	}

	pub fn finish_update(&mut self, ctx: &Context, ui: &mut Ui) {
		if ui.b_back().clicked() {
			self.gui_state.exit();
		}

		if let Some(msg) = self.popup_error.as_deref() {
			let mut open = true;
			Window::new("Error")
				.id(Id::new("multiplayer-error"))
				.open(&mut open)
				.collapsible(false)
				.anchor(Align2::CENTER_CENTER, (0.0, 0.0))
				.resizable(false)
				.show(ctx, |ui| ui.label(msg));

			if !open {
				self.popup_error = None;
			}
		}
	}
}

pub(super) trait CommonGs: GameState {
	fn common(&mut self) -> &mut Common;

	fn disable(&mut self) {}
	fn loop_iter(&mut self, game: &mut Game, frame: &mut GliumFrame) -> bool;
}

impl<T> GameState for T where T: CommonGs {
	fn enable(&mut self, game: &mut Game) {
		self.common().gui_state.enable(&mut game.gui);
	}

	fn push(&mut self, _game: &mut Game, msg: Option<Box<dyn Any>>) {
		debug_assert!(msg.is_none());
	}

	fn disable(&mut self, _game: &mut Game) {
		let common = self.common();
		common.connection_task = None;
		common.popup_error = None;
		CommonGs::disable(self);
	}

	fn action(&mut self, action: ActionEvent) {
		self.common().gui_state.action(action);
	}

	fn event(&mut self, game: &mut Game, event: &Event<()>) {
		self.common().gui_state.event(game, event);
	}

	fn loop_iter(&mut self, game: &mut Game, frame: &mut GliumFrame) -> Status {
		let should_repaint = CommonGs::loop_iter(self, game, frame);

		let common = self.common();
		if common.is_connecting() || should_repaint {
			/*
			 * Want to repaint while waiting as the game could start at any moment
			 * even when there's no user input.
			 */
			common.gui_state.should_repaint();
		}

		if common.gui_state.should_exit() {
			if common.popup_error.is_some() {
				common.popup_error = None;
				Status::Ok
			} else {
				Status::PopState
			}
		} else if let Some(res) = common.connection_finished(&game.tokio_runtime) {
			match res {
				Ok(res) => Status::PushState(GameStateId::Playing, Some(Box::new(res))),
				Err(err) => {
					let err = format!("failed starting multiplayer: {err}");
					log::warn!("{err}");
					common.popup_error = Some(err.into_boxed_str());
					Status::Ok
				}
			}
		} else {
			Status::Ok
		}
	}
}
