// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod local;
pub mod enter_address;
mod common;

use std::any::Any;

use glium::Frame as GliumFrame;
use winit::event::Event;

use crate::game::{Game, config::{self, action::ActionEvent}, state::{GameState, GameStateId, Status}};
use crate::gui::{GuiState, style::UiExt};
use crate::net::{client::Client, serp::PlayRequest};

pub struct PlaySelection {
	gui_state: GuiState,
}

impl PlaySelection {
	pub fn new() -> PlaySelection {
		PlaySelection { gui_state: GuiState::new() }
	}
}

impl GameState for PlaySelection {
	fn enable(&mut self, game: &mut Game) {
		self.gui_state.enable(&mut game.gui);
	}

	fn push(&mut self, _game: &mut Game, msg: Option<Box<dyn Any>>) {
		debug_assert!(msg.is_none());
	}

	fn disable(&mut self, _game: &mut Game) {}

	fn action(&mut self, action: ActionEvent) {
		self.gui_state.action(action);
	}

	fn event(&mut self, game: &mut Game, event: &Event<()>) {
		self.gui_state.event(game, event);
	}

	fn loop_iter(&mut self, game: &mut Game, frame: &mut GliumFrame) -> Status {
		enum Action {
			Local,
			EnterAddress,
			Singleplayer,
		}

		let mut action = None;
		let config = &mut game.config.borrow_mut();

		GuiState::update(&mut game.gui, "play-selection", &game.window, |_, ui| {
			ui.h1("Play").no_bottom_padding().add();
			ui.h2("Multiplayer").add();
			if ui.b2("Local").clicked() {
				action = Some(Action::Local);
			}

			ui.add_space(4.0);
			if ui.b2("Enter Address").clicked() {
				action = Some(Action::EnterAddress);
			}

			ui.h2("Singleplayer").add();
			config::set_changed!(config, ui.checkbox(&mut config.gui.spectate, "Join as Spectator").changed());
			ui.add_space(15.0);

			if ui.b2("Play").clicked() {
				action = Some(Action::Singleplayer);
			}

			if ui.b_back().clicked() {
				self.gui_state.exit();
			}
		});

		game.gui.render(&game.display, frame);

		match action {
			_ if self.gui_state.should_exit() => Status::PopState,
			Some(Action::Local) => Status::PushState(GameStateId::PsLocal, None),
			Some(Action::EnterAddress) => Status::PushState(GameStateId::PsEnterAddress, None),
			Some(Action::Singleplayer) => {
				let request = PlayRequest {
					game_id: Box::from(""),
					spectate: config.gui.spectate,
					style: config.create_style(),
					extension: Box::new([]),
				};
				Status::PushState(GameStateId::Playing, Some(Box::new(Client::singleplayer(request, false))))
			},
			None => Status::Ok,
		}
	}
}
