// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::any::Any;

use glium::Frame as GliumFrame;
use egui::{RichText, Margin, Frame};
use winit::event::Event;

use crate::game::{Game, config::action::ActionEvent, state::{GameState, Status}};
use crate::gui::{GuiState, style::{UiExt, FontSize, H3_SIZE, H3_COLOUR}};
use crate::net::info::{Info, InfoType};

pub struct About {
	gui_state: GuiState,
	info: Info,
}

impl About {
	pub fn new() -> About {
		About {
			gui_state: GuiState::new(),
			info: Info::new(),
		}
	}
}

const ABOUT_TEXT: &str = "Spaceships is a simple free-as-in-freedom multiplayer game created by Marcus Del Favero and written in Rust. Right now the game is very simple, you are a spaceship that can shoot at other spaceships to destroy them and increase your score. Movement is done with WASD or the arrow keys, shooting is done with the left mouse button or spacebar. Periodically the server spawns in ammo crates to supply players with ammo, as well as spawning powerups which give players a temporary effect.\n\nFor more information, see the documentation at https://spaceships.me/docs as well as the attached `README.md`.";

impl GameState for About {
	fn enable(&mut self, game: &mut Game) {
		self.gui_state.enable(&mut game.gui);
	}

	fn push(&mut self, _game: &mut Game, msg: Option<Box<dyn Any>>) {
		debug_assert!(msg.is_none());
	}

	fn disable(&mut self, _game: &mut Game) {}

	fn action(&mut self, action: ActionEvent) {
		self.gui_state.action(action);
	}

	fn event(&mut self, game: &mut Game, event: &Event<()>) {
		self.gui_state.event(game, event);
	}

	fn loop_iter(&mut self, game: &mut Game, frame: &mut GliumFrame) -> Status {
		GuiState::update(&mut game.gui, "about", &game.window, |_, ui| {
			Frame::none().outer_margin(Margin::symmetric(80.0, 0.0)).show(ui, |ui| {
				ui.h1("About").add();
				ui.label(RichText::new(ABOUT_TEXT).font_size(16.0).strong());

				ui.h2("Build Information").add();

				self.info.for_each(|key, value, info_type| {
					ui.horizontal_wrapped(|ui| {
						ui.label(RichText::new(key).font_size(H3_SIZE).color(H3_COLOUR));
						let label_text = RichText::new(value).font_size(H3_SIZE);
						match info_type {
							InfoType::Text => ui.label(label_text.strong()),
							InfoType::Url => ui.hyperlink_to(label_text, value),
						}
					});
				});

				if ui.b_back().clicked() {
					self.gui_state.exit();
				}
			});
		});

		game.gui.render(&game.display, frame);

		if self.gui_state.should_exit() { Status::PopState }
		else { Status::Ok }
	}
}
