// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
mod state;
pub mod style;
pub mod title;
pub mod play_selection;
pub mod settings;
pub mod about;
pub mod layout_job_helper;

pub use state::GuiState;

use std::{fs, path::Path, sync::Arc};

use egui::{ViewportId, FontDefinitions, FontFamily, FontData, FontId, TextStyle, Context};
use egui_glium::EguiGlium;
use glium::{Display, Frame as GliumFrame};
use glutin::surface::WindowSurface;
use winit::{window::Window, event_loop::EventLoop};

use crate::game::filesystem::{Filesystem, FsBase};

pub struct Gui {
	inner: EguiGlium,
	extra_pass: bool,
}

pub const LABEL_SIZE: f32 = 12.0;
pub const SCORE_H1_SIZE: f32 = 20.0;
pub const SCORE_H2_SIZE: f32 = 14.0;

impl Gui {
	pub fn new(display: &Display<WindowSurface>, window: &Window, fs: &Filesystem, event_loop: &EventLoop<()>) -> Gui {
		let inner = EguiGlium::new(ViewportId::ROOT, display, window, event_loop);
		let ctx = inner.egui_ctx();
		let mut fonts = FontDefinitions::empty();
		fonts.font_data.insert(String::from("normal"), FontData::from_owned(fs::read(fs.get(FsBase::Static, Path::new("fonts/inter.otf"))).unwrap()));
		fonts.font_data.insert(String::from("scores"), FontData::from_owned(fs::read(fs.get(FsBase::Static, Path::new("fonts/press_start_2p.ttf"))).unwrap()));

		let scores = Arc::from("scores");
		fonts.families.insert(FontFamily::Proportional, vec![String::from("normal")]);
		fonts.families.insert(FontFamily::Name(Arc::clone(&scores)), vec![String::from("scores")]);

		ctx.options_mut(|options| options.line_scroll_speed = 60.0);

		ctx.set_fonts(fonts);
		ctx.style_mut(|style| {
			style.text_styles = [
				(TextStyle::Heading, FontId::proportional(16.0)),
				(TextStyle::Body, FontId::proportional(LABEL_SIZE)),
				(TextStyle::Button, FontId::proportional(16.0)),
			].into();

			style::set(style);
		});

		Gui { inner, extra_pass: false }
	}

	pub fn update(&mut self, window: &Window, mut add_contents: impl FnMut(&Context)) {
		if self.extra_pass { // Avoids first-frame jitters
			self.extra_pass = false;
			self.inner.egui_ctx().request_discard("");
		}

		self.inner.run(window, &mut add_contents);
	}

	pub fn do_extra_pass(&mut self) {
		self.extra_pass = true;
	}

	pub fn render(&mut self, display: &Display<WindowSurface>, frame: &mut GliumFrame) {
		self.inner.paint(display, frame);
	}
}
