// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::env;
use std::path::{Path, PathBuf};

use super::FsBase;

pub struct Filesystem {
	config_base: Box<Path>,
}

impl Filesystem {
	pub fn build() -> Result<Filesystem, String> {
		/*
		 * NOTE: Using `$XDG_CONFIG_HOME/spaceships` would make more sense for
		 * the base directory, but polluting `~/.config` isn't a problem as this
		 * is for Flatpak.
		 */
		let config_base = env::var("XDG_CONFIG_HOME").map_err(|err| format!("cannot access XDG_CONFIG_HOME, make sure that this binary is being run in Flatpak: {err}"))?;
		Ok(Filesystem { config_base: PathBuf::from(config_base).into_boxed_path() })
	}

	pub fn get(&self, base: FsBase, path: &Path) -> PathBuf {
		match base {
			FsBase::Static => Path::new("/app/data"),
			FsBase::Config => self.config_base.as_ref(),
		}.join(path)
	}
}
