// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod action;
mod action_manager;
pub mod sky;

use std::borrow::Cow;
use std::{fs::File, path::Path};
use std::io::{Read, Write, BufReader, BufWriter, ErrorKind};

use serde::{Serialize, Deserialize};
use ron::{ser::{self, PrettyConfig}, de};
use fs4::fs_std::FileExt;
use nonempty::nonempty;
use glam::Vec3;
use strum_macros::{EnumIter, IntoStaticStr};
use kira::Decibels;

use action::ActionManager;
use sky::{SkyConfig, Gradient, Bezier};

use super::filesystem::{Filesystem, FsBase};

use crate::world::{colour::Colour, player::{PlayerStyle, PlayerNameRef, PlayerNameMut}};
use crate::net::server::PORT;

#[derive(Serialize, Deserialize)]
pub struct Config {
	#[serde(skip)] changed: bool,

	/*
	 * Need to use default to allow upgrading versions without any config errors
	 * (needed this after that Flatpak distribution).
	 */
	#[serde(default = "default_name")] pub name: PlayerNameMut,
	#[serde(default)] pub colour: Colour,
	#[serde(rename = "input_bindings")] pub action_manager: ActionManager,
	#[serde(default)] pub gui: Gui,
	pub graphics: Graphics,
	#[serde(default)] pub sound: Sound,
}

#[derive(Serialize, Deserialize)]
pub struct Gui {
	pub spectate: bool,
	pub host: String,
	pub port: u16,
	pub game_id: String,
}

impl Default for Gui {
	fn default() -> Gui {
		Gui { spectate: false, host: String::new(), port: PORT, game_id: String::new() }
	}
}

#[derive(Serialize, Deserialize)]
pub struct Graphics {
	#[serde(default)] pub sky: Sky,
	#[serde(default)] pub custom_sky: SkyConfig,
	#[serde(default)] pub sky_perf: SkyPerformance,
	pub particles: Particles,
	#[serde(default = "default_forcefield_lightning")] pub forcefield_lightning: bool,
}

impl Graphics {
	pub fn get_sky_config(&self) -> Cow<'_, SkyConfig> {
		match self.sky {
			Sky::Preset(preset) => Cow::Owned(preset.to_config()),
			Sky::Custom => Cow::Borrowed(&self.custom_sky),
		}
	}
}

#[derive(Serialize, Deserialize, PartialEq, Eq)]
pub enum Sky {
	Preset(SkyPreset),
	Custom,
}

impl Default for Sky {
	fn default() -> Sky {
		Sky::Preset(SkyPreset::default())
	}
}

#[derive(Default, Clone, Copy, PartialEq, Eq, Serialize, Deserialize, EnumIter, IntoStaticStr)]
pub enum SkyPreset {
	#[default] Night,
	Galaxy,
	Fire,
	Forest,
	Ocean,
}

impl SkyPreset {
	pub fn to_config(self) -> SkyConfig {
		match self {
			SkyPreset::Night => SkyConfig {
				gradients: [
					Gradient::Bezier(Bezier {
						pre_power: 0.9375,
						points: nonempty![Vec3::ZERO, Vec3::new(0.125, 0.0, 0.5), Vec3::new(0.375, 0.125, 0.75), Vec3::new(0.625, 0.1875, 1.0)],
						post_power: 1.0,
					}),
					Gradient::Bezier(Bezier {
						pre_power: 0.9375,
						points: nonempty![Vec3::ZERO, Vec3::new(0.375, 0.125, 1.0)],
						post_power: 1.0,
					}),
				],
				cloud_brightness: 0.109375,
				star_colour_offset: 0.625,
				.. SkyConfig::default()
			},
			SkyPreset::Galaxy => SkyConfig {
				gradients: [
					Gradient::Bezier(Bezier::new(nonempty![Vec3::ZERO, Vec3::new(0.75, 0.0625, 0.375), Vec3::new(1.0, 0.125, 0.5), Vec3::new(1.0, 0.25, 1.0)])),
					Gradient::Bezier(Bezier::new(nonempty![Vec3::ZERO, Vec3::new(0.375, 0.0625, 0.75), Vec3::new(0.5, 0.125, 1.0), Vec3::new(1.0, 0.25, 1.0)])),
				],
				.. SkyConfig::default()
			},
			SkyPreset::Fire => SkyConfig {
				gradients: [
					Gradient::Bezier(Bezier {
						pre_power: 1.5,
						points: nonempty![Vec3::ZERO, Vec3::new(1.0, 0.0, 0.0), Vec3::new(1.0, 1.0, 0.0), Vec3::new(1.0, 1.0, 0.375)],
						post_power: 1.0,
					}),
					Gradient::Bezier(Bezier {
						pre_power: 1.25,
						points: nonempty![Vec3::ZERO, Vec3::new(1.0, 0.0, 0.0), Vec3::new(1.0, 1.0, 0.0), Vec3::new(1.0, 1.0, 0.375)],
						post_power: 1.0,
					}),
				],
				cloud_brightness: 0.375,
				star_density: 1500.0,
				.. SkyConfig::default()
			},
			SkyPreset::Forest => SkyConfig {
				gradients: [
					Gradient::Bezier(Bezier {
						pre_power: 1.125,
						points: nonempty![Vec3::ZERO, Vec3::new(0.0625, 0.5, 0.125), Vec3::new(0.125, 0.75, 0.25), Vec3::new(0.25, 1.0, 0.5)],
						post_power: 1.0,
					}),
					Gradient::Bezier(Bezier {
						pre_power: 1.125,
						points: nonempty![Vec3::ZERO, Vec3::new(0.125, 0.5, 0.0625), Vec3::new(0.25, 0.75, 0.125), Vec3::new(0.5, 1.0, 0.25)],
						post_power: 1.0,
					}),
				],
				.. SkyConfig::default()
			},
			SkyPreset::Ocean => SkyConfig {
				gradients: [
					Gradient::Bezier(Bezier {
						pre_power: 1.0,
						points: nonempty![Vec3::new(0.0, 0.0, 0.25), Vec3::new(0.0625, 0.03125, 0.5), Vec3::new(0.25, 0.125, 1.0)],
						post_power: 0.875,
					}),
					Gradient::Bezier(Bezier {
						pre_power: 1.0,
						points: nonempty![Vec3::ZERO, Vec3::new(0.0, 0.25, 1.0), Vec3::new(0.0, 1.0, 1.0)],
						post_power: 1.0,
					}),
				],
				cloud_brightness: 0.375,
				star_density: 300.0,
				star_intensity: 2.0,
				star_colour_brightness: 1.0,
				star_colour_offset: 0.5,
			},
		}
	}
}

#[derive(Clone, Serialize, Deserialize)]
pub struct SkyPerformance {
	pub clouds: usize,
	pub stars: bool,
	pub framebuffer: bool,
}

pub const CLOUD_LAYERS: usize = 3;

impl Default for SkyPerformance {
	fn default() -> SkyPerformance {
		SkyPerformance {
			clouds: CLOUD_LAYERS,
			stars: true,
			framebuffer: true,
		}
	}
}

#[derive(Serialize, Deserialize)]
pub struct Particles {
	pub thrust: bool,
	pub powerup: bool,
	pub death: bool,
	pub hit: bool,
	#[serde(default = "default_forcefield_deflection_particles")] pub forcefield_deflection: bool,
}

#[derive(Serialize, Deserialize)]
pub struct Sound {
	pub amplitude: f32,
	pub mute: bool,
}

impl Sound {
	pub fn get_volume(&self) -> Decibels {
		let amplitude = if self.mute { 0.0 } else { self.amplitude };
		let mut volume = 20.0 * amplitude.log10();
		if volume.is_nan() { volume = f32::NEG_INFINITY }
		Decibels(volume.clamp(Decibels::SILENCE.0, Decibels::IDENTITY.0))
	}
}

impl Default for Sound {
	fn default() -> Sound {
		Sound {
			amplitude: 1.0,
			mute: false,
		}
	}
}

const PATH: &str = "config.ron";

impl Config {
	pub fn new(fs: &Filesystem) -> Config {
		Config::load(fs).unwrap_or_else(|err| {
			let ret = Config::default();
			if let Some(err) = err {
				log::warn!("failed loading config: {err}");
				/*
				 * Don't want to save the config file on error, as it might be a
				 * syntax error from manual editing and I don't want to overwrite
				 * the user's changes.
				 */
			} else {
				log::info!("Config file doesn't exist, creating one");
				if let Err(err) = ret.save(fs) {
					log::warn!("cannot save config: {err}");
				}
			}

			ret
		})
	}

	/**
	 * Tries to load the config from the file.
	 *
	 * Returns `Ok` on success, `Err(None)` when the file isn't found (so falling
	 * back to default) or `Err(Some(String))` on another error worth logging.
	 */
	fn load(fs: &Filesystem) -> Result<Config, Option<String>> {
		let path = fs.get(FsBase::Config, Path::new(PATH));
		let file = match File::open(&path) {
			Ok(file) => file,
			Err(err) if err.kind() == ErrorKind::NotFound => return Err(None),
			Err(err) => return Err(Some(format!("cannot open file \"{}\": {err}", path.display()))),
		};
		FileExt::lock_shared(&file).map_err(|err| Some(format!("cannot lock (shared) file: {err}")))?;
		let ret = Config::deserialise(BufReader::new(&file));
		FileExt::unlock(&file).map_err(|err| Some(format!("cannot unlock file: {err}")))?;
		ret
	}

	fn deserialise(reader: impl Read) -> Result<Config, Option<String>> {
		de::from_reader(reader).map_err(|err| Some(format!("failed deserialising: {err}")))
	}

	pub fn save_if_changed(&mut self, fs: &Filesystem) {
		if self.changed {
			/*
			 * If saving fails for whatever reason, try again on the next attempt
			 * at saving rather than marking it as unchanged.
			 */
			if let Err(err) = self.save(fs) {
				log::warn!("cannot save config: {err}");
			}
			self.changed = false;
		}
	}

	fn save(&self, fs: &Filesystem) -> Result<(), String> {
		/*
		 * Cannot call `File::create` because that truncates the file, which
		 * breaks the locking. Instead I am truncating the file but only after
		 * locking it.
		 */
		let path = fs.get(FsBase::Config, Path::new(PATH));
		let file = File::options().write(true).create(true).truncate(false).open(&path).map_err(|err| format!("cannot create file \"{}\": {err}", path.display()))?;
		file.lock_exclusive().map_err(|err| format!("cannot lock (exclusive) file: {err}"))?;
		file.set_len(0).map_err(|err| format!("cannot truncate file: {err}"))?;
		let ret = self.serialise(BufWriter::new(&file));
		FileExt::unlock(&file).map_err(|err| format!("cannot unlock file: {err}"))?;
		ret
	}

	fn serialise(&self, mut writer: impl Write) -> Result<(), String> {
		writeln!(writer, "// Warning: All comments made will be automatically overwritten when changing these in the GUI").map_err(|err| format!("cannot write first comment: {err}"))?;
		ser::to_writer_pretty(&mut writer, self, PrettyConfig::new().indentor(String::from("\t"))).map_err(|err| format!("cannot serialise: {err}"))?;
		writeln!(writer).map_err(|err| format!("cannot write final newline: {err}"))?; // Need to add a final newline
		writer.flush().map_err(|err| format!("cannot flush file: {err}"))
	}

	pub fn changed(&mut self) {
		self.changed = true;
	}

	pub fn set_changed(&mut self, changed: bool) {
		self.changed |= changed;
	}

	pub fn create_style(&self) -> PlayerStyle {
		PlayerStyle {
			name: self.name.to_shared(),
			colour: self.colour,
		}
	}
}

/*
 * Prevents borrow checker problems with `config.set_changed(egui_stuff(&mut
 * config.stuff).changed())`. I don't need this macro but the alternative is too
 * much boilerplate (it's just an extra line, but that adds up to a lot of
 * lines).
 */
macro_rules! set_changed {
	( $config:expr, $x:expr ) => {
		{
			let changed = $x;
			$config.set_changed(changed);
		}
	};
}

pub(crate) use set_changed;

impl Default for Config {
	fn default() -> Config {
		Config {
			changed: false,
			name: default_name(),
			colour: Colour::RED,
			action_manager: ActionManager::default(),
			gui: Gui::default(),
			graphics: Graphics {
				sky: Sky::default(),
				custom_sky: SkyPreset::default().to_config(),
				sky_perf: SkyPerformance::default(),
				particles: Particles { thrust: true, powerup: true, death: true, hit: true, forcefield_deflection: default_forcefield_deflection_particles() },
				forcefield_lightning: default_forcefield_lightning(),
			},
			sound: Sound::default(),
		}
	}
}

fn default_name() -> PlayerNameMut { const { PlayerNameRef::new("Player").unwrap() }.owned() }
fn default_forcefield_deflection_particles() -> bool { true }
fn default_forcefield_lightning() -> bool { true }
