// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
pub mod state;
pub mod config;
pub mod player_gfx;

/**
 * Using different implementations of the Filesystem struct which abstracts
 * access to the filesystem based on how the game is distributed. Right now the
 * distributions released to the average user are the standalone and Flatpak
 * distributions.
 *
 * If the `standalone` flag is set, then this is a version released to the
 * average user. If the `standalone` flag isn't set, then this is used during
 * development.
 *
 * For the standalone distribution, the `data` directory is found relative to
 * the path of the Spaceships binary. This makes things easy to use for the
 * average user as they don't need to worry about their working directory.
 *
 * However, this isn't ideal for use during development as the Spaceships binary
 * is located in `./target/{debug,release}/spaceships`, and I cannot put the
 * data in that directory. Instead for development, the current working
 * directory is used.
 */
pub mod filesystem;

use std::any::Any;
use std::rc::Rc;
use std::cell::RefCell;
use std::num::NonZeroU32;
use std::net::{SocketAddr, IpAddr, Ipv6Addr};
use std::process;

use glium::{Display, backend::glutin::simple_window_builder::GliumEventLoop};
use glutin::{config::GlConfig, display::{GlDisplay, GetGlDisplay}, surface::{WindowSurface, SurfaceAttributesBuilder}, config::ConfigTemplateBuilder, context::{NotCurrentGlContext, ContextAttributesBuilder}};
use winit::{window::{Window, WindowAttributes, Fullscreen}, event_loop::EventLoop, event::{Event, WindowEvent}, dpi::PhysicalSize};
use glutin_winit::DisplayBuilder;
use raw_window_handle::HasWindowHandle;
use quinn::{Endpoint, VarInt};
use tokio::runtime::{Runtime, Builder as RuntimeBuilder};

use state::{GameStateId, StateManager, Status};
use filesystem::Filesystem;
use config::{Config, action::{Action, ActionEvent, ActionState}};
use player_gfx::PlayerGfxCache;

use crate::playing::Playing;
use crate::gui::{Gui, title::Title, play_selection::{PlaySelection, local::PsLocal, enter_address::PsEnterAddress}, settings::Settings, about::About};
use crate::net::{Address, client::Client, serp::PlayRequest, utils::WaitIdleTimeout};
use crate::world::{colour::Colour, player::PlayerName};
use crate::utils::ctrl_c::CtrlCHandler;

pub struct EndpointContainer(Option<Endpoint>);

impl EndpointContainer {
	pub fn get(&mut self, tokio_runtime: &Runtime) -> Result<Endpoint, String> {
		match &self.0 {
			Some(endpoint) => Ok(endpoint.clone()),
			None => {
				let ret = Game::create_endpoint(tokio_runtime)?;
				self.0 = Some(ret.clone());
				Ok(ret)
			},
		}
	}
}

pub struct Game {
	pub window: Window,
	pub display: Display<WindowSurface>,
	pub gui: Gui,
	pub tokio_runtime: Runtime,
	pub endpoint: EndpointContainer,
	pub fs: Filesystem,
	pub config: Rc<RefCell<Config>>,
	pub player_gfx: PlayerGfxCache,
}

impl Game {
	fn create_tokio_runtime() -> Runtime {
		RuntimeBuilder::new_multi_thread()
			.worker_threads(1)
			.enable_all()
			.build()
			.unwrap()
	}

	fn create_fs_and_config() -> Result<(Filesystem, Rc<RefCell<Config>>), String> {
		let fs = Filesystem::build()?;
		let config = Rc::new(RefCell::new(Config::new(&fs)));
		Ok((fs, config))
	}

	pub fn title() -> Result<(), String> {
		let (fs, config) = Game::create_fs_and_config()?;
		Game::run(GameStateId::Title, None, Game::create_tokio_runtime(), None, fs, config)
	}

	pub fn singleplayer(spectate: bool, name: Option<PlayerName>, colour: Option<Colour>, sanity_checks: bool) -> Result<(), String> {
		let (fs, config) = Game::create_fs_and_config()?;
		let request = PlayRequest {
			game_id: Box::from(""),
			spectate,
			style: config.borrow().create_style().override_with(name, colour),
			extension: Box::new([]),
		};
		Game::run(GameStateId::Playing, Some(Box::new(Client::singleplayer(request, sanity_checks))), Game::create_tokio_runtime(), None, fs, config)
	}

	pub fn multiplayer(addr: Address, game_id: Box<str>, spectate: bool, name: Option<PlayerName>, colour: Option<Colour>, sanity_checks: bool) -> Result<(), String> {
		let (fs, config) = Game::create_fs_and_config()?;
		let tokio_runtime = Game::create_tokio_runtime();
		let endpoint = Game::create_endpoint(&tokio_runtime);
		let request = PlayRequest {
			game_id,
			spectate,
			style: config.borrow().create_style().override_with(name, colour),
			extension: Box::new([]),
		};
		let res = tokio_runtime.block_on(Client::multiplayer(endpoint.clone(), addr, request, sanity_checks))?;
		Game::run(GameStateId::Playing, Some(Box::new(res)), tokio_runtime, Some(endpoint?), fs, config)
	}

	fn build_window(event_loop: &EventLoop<()>) -> (Window, Display<WindowSurface>) { // Copied from SimpleWindowBuilder::build as I want multisampling enabled
		let attributes = WindowAttributes::default().with_title("Spaceships").with_inner_size(PhysicalSize::new(1280, 720));

		// First we start by opening a new Window
		let display_builder = DisplayBuilder::new().with_window_attributes(Some(attributes));
		let config_template_builder = ConfigTemplateBuilder::new().with_stencil_size(1);
		let (window, gl_config) = event_loop.build(display_builder, config_template_builder, |configs| {
			/*
			 * Behaviour differing from glium's SimpleWindowBuilder::build method by
			 * choosing the configuration with the most samples.
			 *
			 * I got inspiration that I could do this from:
			 * 	https://github.com/rust-windowing/glutin/blob/master/glutin_examples/examples/egl_device.rs
			 */
			configs.max_by_key(GlConfig::num_samples).unwrap()
		}).unwrap();

		let window = window.unwrap();

		// Now we get the window size to use as the initial size of the Surface
		let (width, height): (u32, u32) = window.inner_size().into();
		let attrs = SurfaceAttributesBuilder::<WindowSurface>::new()
			.build(window.window_handle().expect("couldn't obtain raw window handle").into(), NonZeroU32::new(width).unwrap(), NonZeroU32::new(height).unwrap());

		// Finally we can create a Surface, use it to make a PossiblyCurrentContext and create the glium Display
		// SAFETY: Copied from glium's SimpleWindowBuilder, should be fine
		let surface = unsafe { gl_config.display().create_window_surface(&gl_config, &attrs).unwrap() };
		let context_attributes = ContextAttributesBuilder::new().build(Some(window.window_handle().expect("couldn't obtain raw window handle").into()));

		// SAFETY: Copied from glium's SimpleWindowBuilder, should be fine
		let current_context = unsafe {
			gl_config.display().create_context(&gl_config, &context_attributes).expect("failed to create context")
		}.make_current(&surface).unwrap();

		let display = Display::from_context_surface(current_context, surface).unwrap();

		(window, display)
	}

	pub fn run(
		init_state: GameStateId, arg_to_push: Option<Box<dyn Any>>, tokio_runtime: Runtime, endpoint: Option<Endpoint>,
		fs: Filesystem, config: Rc<RefCell<Config>>,
	) -> Result<(), String> {
		let event_loop = EventLoop::new().unwrap();
		let ctrl_c_handler = CtrlCHandler::new();
		{
			let ctrl_c_handler = ctrl_c_handler.clone();
			let sender = event_loop.create_proxy();

			/*
			 * Wakes up the event loop with a useless user event to check the
			 * Ctrl+C handler and then exit.
			 *
			 * This is needed if in the GUI and the window hasn't been focused
			 * within a second (or whatever the time is set to), as no events
			 * arrive during that time (as the GUI doesn't need to always be
			 * rendering).
			 */
			tokio_runtime.spawn(async move { ctrl_c_handler.listen_with(|| _ = sender.send_event(())).await });
		}

		let (window, display) = Game::build_window(&event_loop);
		let gui = Gui::new(&display, &window, &fs, &event_loop);

		let mut game = Game { window, display, gui, tokio_runtime, endpoint: EndpointContainer(endpoint), fs, config, player_gfx: PlayerGfxCache::default() };
		let mut stack = vec![init_state];

		let mut sman = StateManager::new();
		let mut enable = true;
		let mut push_msg = Some(arg_to_push);

		#[allow(deprecated)] // Refactoring is too much work
		event_loop.run(|event, window_target| {
			let Some(id) = stack.last() else {
				window_target.exit();
				stack.clear();
				return;
			};

			let state = sman.get(*id, || match id {
				GameStateId::Title => Box::new(Title::new()),
				GameStateId::PlaySelection => Box::new(PlaySelection::new()),
				GameStateId::PsLocal => Box::new(PsLocal::new()),
				GameStateId::PsEnterAddress => Box::new(PsEnterAddress::new(&game.config.borrow())),
				GameStateId::Playing => Box::new(Playing::new(&mut game)),
				GameStateId::Settings => Box::new(Settings::new(&mut game)),
				GameStateId::About => Box::new(About::new()),
			});

			if enable {
				if let Some(msg) = push_msg.take() {
					state.push(&mut game, msg);
				}
				state.enable(&mut game);
				enable = false;
			} else {
				assert!(push_msg.is_none());
			}

			if ctrl_c_handler.should_exit() {
				game.config.borrow_mut().save_if_changed(&game.fs);
				state.disable(&mut game);
				window_target.exit();
				stack.clear();
				return;
			}

			state.event(&mut game, &event);

			if let Event::WindowEvent { event, .. } = &event {
				match event {
					WindowEvent::CloseRequested => {
						game.config.borrow_mut().save_if_changed(&game.fs);
						state.disable(&mut game);
						window_target.exit();
						stack.clear();
						return;
					},
					WindowEvent::Resized(size) => game.display.resize((*size).into()),
					WindowEvent::RedrawRequested => {
						let mut frame = game.display.draw();

						match state.loop_iter(&mut game, &mut frame) {
							Status::Ok => (),
							Status::PushState(id, args) => {
								game.config.borrow_mut().save_if_changed(&game.fs);
								state.disable(&mut game);
								stack.push(id);
								push_msg = Some(args);
								enable = true;
							},
							Status::PopState => {
								game.config.borrow_mut().save_if_changed(&game.fs);
								state.disable(&mut game);
								stack.pop();
								enable = true;
							},
						}
						game.window.pre_present_notify(); // Apparently makes things better on Wayland, though I don't notice any difference
						frame.finish().unwrap();
						return; // Don't want to propagate this to the state's event method
					},
					_ => (),
				}

				game.config.borrow_mut().action_manager.event(event, |action| {
					if matches!(action, ActionEvent::Action(Action::ToggleFullscreen, ActionState::Pressed)) {
						game.window.set_fullscreen(if game.window.fullscreen().is_some() { None } else { Some(Fullscreen::Borderless(None)) });
					} else {
						state.action(action);
					}
				});
			}
		}).unwrap();

		game.finish(); // Don't think it's possible to get here
	}

	fn finish(self) -> ! {
		drop(self.gui); // Calls a segfault for whatever unsound reason if dropped after
		drop(self.display);
		drop(self.window);

		if let Some(endpoint) = &self.endpoint.0 {
			self.tokio_runtime.block_on(async {
				endpoint.close(VarInt::from_u32(0), &[]);
				endpoint.wait_idle_timeout().await;
			});
		}

		process::exit(0);
	}

	fn create_endpoint(tokio_runtime: &Runtime) -> Result<Endpoint, String> {
		tokio_runtime.block_on(async { Endpoint::client(SocketAddr::new(IpAddr::V6(Ipv6Addr::UNSPECIFIED), 0)) }).map_err(|err| format!("cannot create QUIC client endpoint: {err}"))
	}
}
