// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use std::num::NonZeroU32;

use glam::UVec2;
use serde::{Serialize, Deserialize};

use super::{Blocks, MAX_BLOCKS};

#[derive(Clone, Serialize, Deserialize)]
pub struct CompressedBlocks {
	block_size: NonZeroU32,
	grid_size: UVec2,
	data: Box<[u8]>,
}

impl From<&Blocks> for CompressedBlocks {
	fn from(blocks: &Blocks) -> CompressedBlocks {
		let mut data = Vec::new();

		let mut prev = false;
		let mut count = 0usize;
		for &is_block in &blocks.data {
			if is_block != prev {
				while count > u8::MAX as usize {
					data.push(u8::MAX);
					data.push(0);
					count -= u8::MAX as usize;
				}
				data.push(count as u8);

				count = 0;
				prev = is_block;
			}

			count += 1;
		}

		CompressedBlocks {
			block_size: blocks.block_size,
			grid_size: blocks.grid_size,
			data: data.into_boxed_slice(),
		}
	}
}

impl TryFrom<CompressedBlocks> for Blocks {
	type Error = &'static str;

	fn try_from(blocks: CompressedBlocks) -> Result<Blocks, Self::Error> {
		let expected_len = blocks.grid_size.x.saturating_mul(blocks.grid_size.y) as usize;
		if expected_len > MAX_BLOCKS { return Err("too many blocks"); }

		let mut data = Vec::with_capacity(expected_len);

		let mut is_block = false;
		for count in blocks.data {
			for _ in 0..count {
				data.push(is_block);
			}

			if data.len() > expected_len {
				return Err("too many elements");
			}

			is_block = !is_block;
		}

		for _ in 0..expected_len - data.len() {
			data.push(is_block);
		}

		debug_assert_eq!(data.len(), expected_len);

		Ok(Blocks {
			block_size: blocks.block_size,
			grid_size: blocks.grid_size,
			data: data.into_boxed_slice(),
		})
	}
}

#[cfg(test)]
mod tests {
	use super::*;

	#[test]
	fn all_same() {
		fn get(grid_size: UVec2, block: bool) -> Blocks {
			Blocks {
				block_size: 5.try_into().unwrap(),
				grid_size,
				data: vec![block; grid_size.element_product() as usize].into_boxed_slice(),
			}
		}

		for _ in 0..100 {
			let size = rand::random::<UVec2>() % 100;
			test(get(size, true));
			test(get(size, false));
		}
	}

	#[test]
	fn empty() {
		test(Blocks {
			block_size: 5.try_into().unwrap(),
			grid_size: UVec2::ZERO,
			data: Box::new([]),
		});
	}

	#[test]
	fn white_noise() {
		fn get(grid_size: UVec2, prob: f32) -> Blocks {
			Blocks {
				block_size: 5.try_into().unwrap(),
				grid_size,
				data: (0..grid_size.element_product() as usize).map(|_| rand::random::<f32>() < prob).collect(),
			}
		}

		for _ in 0..500 {
			test(get(rand::random::<UVec2>() % 100, rand::random::<f32>()));
		}
	}

	#[test]
	fn random_runs() {
		for _ in 0..500 {
			let grid_size = rand::random::<UVec2>() % 100;
			let expected_len = grid_size.element_product() as usize;
			let mut data = Vec::with_capacity(expected_len);

			while data.len() < expected_len {
				let count = rand::random::<u32>() % 500;
				let is_block = rand::random::<bool>();
				for _ in 0..count.min((expected_len - data.len()) as u32) {
					data.push(is_block);
				}
			}

			test(Blocks {
				block_size: 5.try_into().unwrap(),
				grid_size,
				data: data.into_boxed_slice(),
			});
		}
	}

	fn test(blocks: Blocks) {
		let blocks2 = Blocks::try_from(CompressedBlocks::from(&blocks)).unwrap();
		assert!(blocks.block_size == blocks2.block_size && blocks.grid_size == blocks2.grid_size && blocks.data == blocks2.data);
	}
}
