// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
#[cfg(feature = "client")] mod renderer;
#[cfg(feature = "client")] mod mesh;
mod range_iter;
pub mod compressed;
#[cfg(feature = "client")] mod border;

#[cfg(feature = "client")] pub use renderer::Renderer as BlocksRenderer;
pub use range_iter::RangeIterator;

use std::num::NonZeroU32;

use glam::{UVec2, Vec2};
#[cfg(feature = "client")] use glam::IVec2;

pub struct Blocks {
	block_size: NonZeroU32,
	grid_size: UVec2,
	data: Box<[bool]>,
}

pub const MAX_BLOCKS: usize = 1048576;

impl Blocks {
	pub fn new(block_size: NonZeroU32, grid_size: UVec2, data: Box<[bool]>) -> Blocks {
		Blocks { block_size, grid_size, data }
	}

	#[cfg(feature = "client")]
	pub fn world_to_block_pos(&self, world_pos: Vec2) -> (IVec2, bool) {
		let block_size = self.block_size.get() as f32;
		let off = block_size * self.grid_size.as_vec2() / 2.0;
		let block_pos = ((world_pos + off) / block_size).floor().as_ivec2();
		let in_bounds = block_pos.x >= 0 && (block_pos.x as u32) < self.grid_size.x && block_pos.y >= 0 && (block_pos.y as u32) < self.grid_size.y;
		(block_pos, in_bounds)
	}

	/**
	 * Returns an iterator over all blocks colliding with the square centred at
	 * `pos` with radius (half side length) of `radius`.
	 */
	pub fn iter_range(&self, pos: Vec2, radius: f32) -> RangeIterator {
		debug_assert!(radius > 0.0);
		RangeIterator::new(self, pos - Vec2::splat(radius), pos + Vec2::splat(radius))
	}
}
