#!/usr/bin/env python3
# Analyses the performance logs
import sys
import math

SUFFIX = " μs"
MUL = 1e6
INDENT = " " * 3

def secs_to_time(secs):
	return f"{secs * MUL:.3f}{SUFFIX}"

client_count = 0
period = None

all_times = {}

for line in sys.stdin:
	parts = line.strip().split(",")
	target = parts[0]
	match target:
		case "client count":
			client_count = int(parts[1])
		case "period":
			period = float(parts[1])
		case _:
			all_times.setdefault(target, {}).setdefault(client_count, []).append(float(parts[1]))

for target, times_per_client_count in sorted(all_times.items()):
	print(f"\x1b[32m{target}, \x1b[36mx{sum(len(t) for t in times_per_client_count.values())}\x1b[0m")
	for client_count, times in sorted(times_per_client_count.items()):
		mean_t = sum(times) / len(times)
		stddev_t = math.sqrt(sum((mean_t - t) ** 2 for t in times) / len(times)) # Biased but good enough as I don't want to divide by zero
		min_t = min(times)
		max_t = max(times)
		print(f"{INDENT}\x1b[33clients = {client_count}, \x1b[36mx{len(times)}\x1b[0m")
		print(f"{INDENT * 2}mean = {secs_to_time(mean_t)}")
		print(f"{INDENT * 2}stddev = {secs_to_time(stddev_t)}")
		print(f"{INDENT * 2}min = {secs_to_time(min_t)}")
		print(f"{INDENT * 2}max = {secs_to_time(max_t)}")
