// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use rand_chacha::ChaCha20Rng;
use glam::Vec2;

use super::line::Line;

use crate::utils::Float;

pub struct LightningConfig {
	pub min_split_line_length: f32,
	pub pos_mul: f32,
	pub t_range: f32,
}

impl Default for LightningConfig {
	fn default() -> LightningConfig {
		LightningConfig {
			min_split_line_length: 0.1,
			pos_mul: 0.3,
			t_range: 0.2
		}
	}
}

pub fn lightning(line: Line, rng: &mut ChaCha20Rng, config: &LightningConfig, out: &mut Vec<Line>) {
	let length = line.0.distance(line.1);
	if length < config.min_split_line_length {
		out.push(line);
		return;
	}

	let t = rng.next_f32() * config.t_range + 0.5 - config.t_range / 2.0;
	let p = line.0.lerp(line.1, t) + Vec2::new(rng.next_f32() * 2.0 - 1.0, rng.next_f32() * 2.0 - 1.0) * length * config.pos_mul;

	lightning(Line(line.0, p), rng, config, out);
	lightning(Line(p, line.1), rng, config, out);
}
