// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use png::ColorType;

use super::image::Image;
use super::inkscape;

use crate::utils;

const INPUT: &str = "data/player.svg";
const OUTPUT: &str = "../data/textures/player.png";

pub fn generate() {
	const SIZE: usize = 256;

	if utils::newer(INPUT, OUTPUT) {
		inkscape::gen_grey_layers("player", INPUT, &["base", "colour"], SIZE, SIZE / 2).save(OUTPUT, ColorType::GrayscaleAlpha);
	}
}

pub fn generate_icon(size: usize, path: &str) {
	let input = inkscape::gen_grey_layers("player", INPUT, &["base", "colour", "alpha"], size, size / 2);

	let mut output = Image::new(size, size, 4);
	let player_colour = [255, 63, 31];

	assert!(input.width == size && input.height == size / 2);

	for y in 0..size / 2 {
		for x in 0..size {
			let i = x + y * size;
			let base = input.pixels[i * 3];
			let colour = input.pixels[i * 3 + 1];
			let alpha = input.pixels[i * 3 + 2];
			let t = colour as f64 / 255.0;

			let out_indices = [
				x + y * size,
				x + (size - 1 - y) * size,
			];

			for out_i in out_indices {
				for (i, &colour) in player_colour.iter().enumerate() {
					output.pixels[out_i * 4 + i] = (base as i32 + ((colour - base as i32) as f64 * t) as i32) as u8;
				}

				output.pixels[out_i * 4 + 3] = alpha;
			}
		}
	}

	output.save(path, ColorType::Rgba);
}
