// Copyright Marcus Del Favero 2025
// Licensed under the GNU AGPLv3 with an exception, see `README.md` for details
use crate::utils;

use super::sampler::Sampler;

pub trait Resampler {
	fn next(&mut self, dt: f32) -> f32;
}

pub struct SmoothstepResampler<S> {
	sampler: S,
	x0: f32,
	x1: f32,
	t: f32,
}

impl<S> SmoothstepResampler<S> where S: Sampler {
	pub fn new(mut sampler: S) -> SmoothstepResampler<S> {
		let (x0, x1) = (sampler.next(), sampler.next());
		SmoothstepResampler { sampler, x0, x1, t: 0.0 }
	}
}

impl<S> Resampler for SmoothstepResampler<S> where S: Sampler {
	fn next(&mut self, dt: f32) -> f32 {
		self.t += dt;

		#[allow(clippy::while_float)]
		while self.t >= 1.0 {
			self.x0 = self.x1;
			self.x1 = self.sampler.next();
			self.t -= 1.0;
		}

		utils::lerp(self.x0, self.x1, utils::smoothstep(self.t))
	}
}
